#!/usr/bin/env python
# -*- coding: utf-8 -*- 

# wg: http://mein-neues-blog.de/2012/01/01/gimp-plug-in-moderated-blur-for-tilt-shift-or-depth-of-field-effects/  
# „Moderated Blur“ for Tilt Shift or Depth-Of-Field Effects
# https://en.wikipedia.org/wiki/Tilt-shift_photography#Selective_focus

# Moderator: Wybieramy warstwę, która będzie konwertowana na Skalę szarości i zostanie wykorzystana jako "mapa rozmycia" 
# (decyduje o zależności lub braku zależności pomiędzy zmiennymi; warstwa, która kieruje zakresem oddziaływania procesu).
# Min Blur: minimalne rozmycie, stosowane w czarnych obszarach (regionach) mapy
# Max Blur: maksymalne rozmycie, stosowane w białych obszarach mapy
# Accuracy: (Dokładność) wartość w różnych regionach. Każdy region jest rozmywany niezależnie, co zwiększa czas renderowania.
# Order: (zamówienie) jaką intensywność rozmycia należy wkomponować na szczycie. Użyj Min dla ostrych krawędzi i Max dla łagodnych przejść
# Scale: (skala) używać skali liniowej lub logarytmicznej promieni rozmycia. Logarytmiczne rozmycie startuje łagodniej i tworzy łagodniejsze pole głębi.   
#
# Tworzymy warstwę z zawartością szarości, gdzie obszary białe spowoduje największy promień rozmycia a czarne będzie powodował min. rozmycie.
# Przesuwamy ją wszędzie gdzie chcemy w stosie warstw, można nawet ustawić jej krycie na 0.
# Wybierz warstwę do zastosowania efektu tilt-shift
# Uruchom wtyczkę Moderated Blur
# Wybierz utworzoną warstwę moderator, wprowadź swoje ustawienia i rozpocznij proces.
# Później opcjonalnie można usunąć warstwę moderator.

 
import math
from gimpfu import *
import time

def python_fu_moderated_blur(img, draw, moderator, minBlur, maxBlur, accuracy, order, log):
    pdb.gimp_image_undo_group_start(img) # początek procedury, która pozwoli nam anulować wszystkie działania naszego skryptu 
    act = pdb.gimp_image_get_active_layer(img) # szczegóły patrz pdb
    # Tworzy nowy obraz od mod z mod jako nową warstwę create a new image from mod with mod as new layer
    pdb.gimp_edit_copy(moderator) # kopiuje moderatora do bufora
    temp = pdb.gimp_image_new(moderator.width, moderator.height, 1) # Tworzy nowy obraz tymczasowy
    mod = pdb.gimp_layer_new(temp, moderator.width, moderator.height, 3, "moderator", 100, 0) # Tworzy nową warstwę tymczasową
    temp.add_layer(mod, 0) # dodaj warswę do obrazu
    floating = pdb.gimp_edit_paste(mod, 0) # wkleja moderatora do nowego obrazu
    pdb.gimp_floating_sel_attach(floating, mod) # wkleja dodając pływającą
    gimp.displays_flush() # Aktualizujemy, odświerzamy przetwarzany obraz na ekranie po zmianach
    # budujemy mapę kolorów z ustawieniem dokładności (Accuracy) 
    colorMap = (0, 0, 0) # Mapa kolorów - krotka z trójką wartości kolorów, zaczynając od czerni
    colors = [0] # wewnętrzna tablica z używanymi kolorami
    # wypełnij colorMap
    if accuracy > 2:
        for i in range(2, int(accuracy)):
            s = int(math.floor((255.0 / (accuracy - 1)) * (i - 1))) # tworzenie brakujących kolorów pmiędzy czarnym i białym
            colorMap += (s, s, s)
            colors.append(s)
    colorMap += (255, 255, 255) # dodanie białego do kolorów
    colors.append(255)
    # indeksowanie obrazu tymczasowego 
    pdb.gimp_convert_indexed(temp, 0, 0, accuracy, 0, 0, "")
    pdb.gimp_image_set_colormap(temp, accuracy * 3, colorMap)
    # kopiowanie powrotne mod do oryginału
    pdb.gimp_edit_copy(mod) # kopiowanie mod do bufora
    mod = pdb.gimp_layer_new(img, moderator.width, moderator.height, 0, "moderator", 100, 0) # Tworzy nową warstwę mod
    img.add_layer(mod, 0) # dodaje nową warstwę mod do oryginalnego obrazu
    pdb.gimp_layer_set_offsets(mod, moderator.offsets[0], moderator.offsets[1]) # przesunąć do oryginalnej pozycji
    floating = pdb.gimp_edit_paste(mod, 0) # wkleja bufor do nowej warstwy mod 
    pdb.gimp_floating_sel_attach(floating, mod) # wklej dodając pływającą
    # proces
    count = 0
    if order == 0:
        colors = reversed(colors)
    for c in colors:
        # oblicz rozmycie
        if log:
            blur = (maxBlur - minBlur) * ((c * c) / (256.0 * 256.0)) + minBlur
        else:
            blur = (maxBlur - minBlur) * (c / 256.0) + minBlur
        # wybierz kolor 
        pdb.gimp_by_color_select(mod, (c, c, c), 0, 2, False, 0, 0, 0)
        # zaznaczenie rosnące
        pdb.gimp_selection_grow(img, math.ceil(blur / 2.0))
        # odwróć zaznaczenie
        pdb.gimp_selection_invert(img)
        # skopiuj obszar do act 
        region = pdb.gimp_layer_copy(act, 1)
        img.add_layer(region, 0)
        # wyczyść obszar 
        pdb.gimp_edit_clear(region)
        # usuń zaznaczenie
        pdb.gimp_selection_clear(img)
        # rozmyj warstwę 
        if(blur > 0):
            pdb.plug_in_gauss_iir2(img, region, blur, blur)
        # w razie potrzeby połącz
        if count > 0:
            pdb.gimp_image_merge_down(img, region, 0)
        count += 1
        gimp.displays_flush() # po zmianach odświeżamy przetwarzany obraz na ekranie 
    # usuń mod
    pdb.gimp_image_remove_layer(img, mod)
    pdb.gimp_image_delete(temp)
    pdb.gimp_image_undo_group_end(img)
    
register(
	"python_fu_moderated_blur",
	"Moderated Blur",
	"""Rozmywa zaznaczoną warstwę według mapy szarości. Proces wykorzystuje różne strefy do mocy rozmycia, 
	   które finalnie są łączone do nowej warstwy.
	Dokładność określa wielkość obszaru (regionu)
	Zamówienie określa jaka intensywność rozmycia warstwy jest komponowana na szczycie
	Skala log używa innych promieni rozmycia niż skala liniowa i ma bardziej miękkie przejścia między Min i Max""",
	"Markus Schmidt <schmidt@boomshop.net>",
	"Grandpa",
	"2011-12-31",
	"<Image>/TEST/Moderated Blur",
	"RGB*, GRAY*",
	[
	    (PF_LAYER, "moderator", "Moderator", None),
		(PF_SLIDER, "minBlur", "Min Blur", 0, (0, 150, 1)),
		(PF_SLIDER, "maxBlur", "Max Blur", 25, (0, 150, 1)),
		(PF_SLIDER, "accuracy", "Accuracy", 5, (2, 255, 1)),
		(PF_SLIDER, "order", "Top Layer (0=Min / 1=Max)", 1, (0, 1, 1)),
		(PF_SLIDER, "log", "Skala (0=Lin / 1=Log)", 1, (0, 1, 1))
	],
	[],
	python_fu_moderated_blur)

main()
